<?php
require_once "auth.php"; // <--- ESTO DEBE SER LA LÍNEA 1
if (session_status() === PHP_SESSION_NONE) { session_start(); }
require_once '../config/db.php'; 

try {
    // --- 1. CÁLCULO DE CAPITAL Y TOTALES ---
    $totalCapital = $pdo->query("SELECT SUM(price) FROM products WHERE status = 'active'")->fetchColumn() ?: 0;
    $totalItems = $pdo->query("SELECT COUNT(*) FROM products")->fetchColumn() ?: 0;
    $totalMarcas = $pdo->query("SELECT COUNT(*) FROM brands")->fetchColumn() ?: 0;
    $totalCategorias = $pdo->query("SELECT COUNT(*) FROM categories")->fetchColumn() ?: 0;
    $totalEquipo = $pdo->query("SELECT COUNT(*) FROM team_members")->fetchColumn() ?: 0;
    
    // Precio Promedio por producto
    $precioPromedio = ($totalItems > 0) ? ($totalCapital / $totalItems) : 0;

    // --- 2. ANÁLISIS DE CATEGORÍAS ---
    $catDensity = $pdo->query("SELECT c.name, COUNT(p.id) as qty, SUM(p.price) as total_val 
                                FROM categories c 
                                JOIN products p ON c.id = p.category_id 
                                GROUP BY c.id ORDER BY total_val DESC LIMIT 6")->fetchAll();

    // --- 3. SEGMENTACIÓN DE MERCADO ---
    $ranges = $pdo->query("SELECT 
        SUM(CASE WHEN price < 500 THEN 1 ELSE 0 END) as econ,
        SUM(CASE WHEN price BETWEEN 500 AND 2500 THEN 1 ELSE 0 END) as inter,
        SUM(CASE WHEN price > 2500 THEN 1 ELSE 0 END) as pro
        FROM products")->fetch();

    // --- 4. ÚLTIMOS MOVIMIENTOS ---
    $recent = $pdo->query("SELECT p.name, p.price, b.name as bname, p.created_at 
                            FROM products p JOIN brands b ON p.brand_id = b.id 
                            ORDER BY p.created_at DESC LIMIT 5")->fetchAll();

} catch (PDOException $e) {
    die("Error crítico: " . $e->getMessage());
}

include 'includes/header.php';
include 'includes/sidebar.php';
?>

<div class="flex-1 flex flex-col h-full overflow-hidden bg-[#f8f9fc] dark:bg-[#050505]">
    <?php include 'includes/navbar.php'; ?>

    <main class="flex-1 overflow-x-hidden overflow-y-auto p-4 md:p-8">
        
        <div class="flex flex-col md:flex-row justify-between items-end mb-10 gap-4">
            <div>
                <span class="text-brand-red font-black text-xs uppercase tracking-[0.3em]">JFD Importaciones</span>
                <h1 class="text-4xl font-display font-black text-gray-900 dark:text-white tracking-tighter">
                    BUSINESS <span class="text-brand-red">INSIGHTS</span>
                </h1>
            </div>
            <div class="flex items-center gap-4 bg-white dark:bg-brand-dark p-4 rounded-2xl shadow-sm border border-gray-100 dark:border-white/5">
                <div class="text-right">
                    <p class="text-[10px] font-black text-gray-400 uppercase">Inversión en Stock</p>
                    <p class="text-2xl font-display font-black text-gray-800 dark:text-white">S/ <?php echo number_format($totalCapital, 2); ?></p>
                </div>
                <div class="w-12 h-12 bg-brand-red rounded-2xl flex items-center justify-center text-white text-2xl shadow-lg shadow-brand-red/30">
                    <i class="ri-bank-card-2-line"></i>
                </div>
            </div>
        </div>

        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-10">
            
            <div class="bg-white dark:bg-brand-dark p-6 rounded-[2rem] shadow-sm border border-gray-100 dark:border-white/5 group">
                <p class="text-[10px] font-black text-gray-400 uppercase mb-2">Marcas Representadas</p>
                <div class="flex items-center justify-between">
                    <h3 class="text-4xl font-display font-black text-gray-800 dark:text-white"><?php echo $totalMarcas; ?></h3>
                    <div class="w-10 h-10 rounded-xl bg-blue-50 dark:bg-blue-500/10 flex items-center justify-center text-blue-500">
                        <i class="ri-shield-star-fill text-xl"></i>
                    </div>
                </div>
                <p class="text-[10px] text-gray-500 mt-4 font-bold uppercase tracking-tight">Portafolio de proveedores</p>
            </div>

            <div class="bg-white dark:bg-brand-dark p-6 rounded-[2rem] shadow-sm border border-gray-100 dark:border-white/5">
                <p class="text-[10px] font-black text-gray-400 uppercase mb-2">Líneas de Negocio</p>
                <div class="flex items-center justify-between">
                    <h3 class="text-4xl font-display font-black text-gray-800 dark:text-white"><?php echo $totalCategorias; ?></h3>
                    <div class="w-10 h-10 rounded-xl bg-yellow-50 dark:bg-yellow-500/10 flex items-center justify-center text-yellow-600">
                        <i class="ri-folders-fill text-xl"></i>
                    </div>
                </div>
                <p class="text-[10px] text-gray-500 mt-4 font-bold uppercase tracking-tight">Categorías Activas</p>
            </div>

            <div class="bg-white dark:bg-brand-dark p-6 rounded-[2rem] shadow-sm border border-gray-100 dark:border-white/5">
                <p class="text-[10px] font-black text-gray-400 uppercase mb-2">Valor Promedio Unidad</p>
                <div class="flex items-center justify-between">
                    <h3 class="text-2xl font-display font-black text-gray-800 dark:text-white">S/ <?php echo number_format($precioPromedio, 0); ?></h3>
                    <div class="w-10 h-10 rounded-xl bg-green-50 dark:bg-green-500/10 flex items-center justify-center text-green-500">
                        <i class="ri-scales-3-line text-xl"></i>
                    </div>
                </div>
                <p class="text-[10px] text-gray-500 mt-4 font-bold uppercase tracking-tight">Ticket medio por ítem</p>
            </div>

            <div class="bg-white dark:bg-brand-dark p-6 rounded-[2rem] shadow-sm border border-gray-100 dark:border-white/5">
                <p class="text-[10px] font-black text-gray-400 uppercase mb-2">Asesores Comerciales</p>
                <div class="flex items-center justify-between">
                    <h3 class="text-4xl font-display font-black text-gray-800 dark:text-white"><?php echo $totalEquipo; ?></h3>
                    <div class="w-10 h-10 rounded-xl bg-purple-50 dark:bg-purple-500/10 flex items-center justify-center text-purple-500">
                        <i class="ri-user-voice-fill text-xl"></i>
                    </div>
                </div>
                <p class="text-[10px] text-gray-500 mt-4 font-bold uppercase tracking-tight">Equipo JFD Import</p>
            </div>
        </div>

        <div class="grid grid-cols-1 lg:grid-cols-12 gap-8 mb-10">
            <div class="lg:col-span-5 bg-white dark:bg-brand-dark p-8 rounded-[2.5rem] shadow-sm border border-gray-100 dark:border-white/5">
                <h4 class="font-bold text-gray-800 dark:text-white uppercase text-xs tracking-widest mb-8 text-center">Densidad de Modelos por Categoría</h4>
                <div class="h-72">
                    <canvas id="radarBusiness"></canvas>
                </div>
            </div>

            <div class="lg:col-span-7 bg-white dark:bg-brand-dark rounded-[2.5rem] shadow-sm border border-gray-100 dark:border-white/5 overflow-hidden">
                <div class="p-8 border-b border-gray-50 dark:border-white/5 flex justify-between items-center bg-gray-50/50 dark:bg-transparent">
                    <h4 class="font-bold text-gray-800 dark:text-white uppercase text-xs tracking-widest italic">Peso Económico por Categoría</h4>
                    <i class="ri-bar-chart-box-line text-gray-400 text-xl"></i>
                </div>
                <div class="overflow-x-auto">
                    <table class="w-full text-left">
                        <thead class="text-[10px] font-black text-gray-400 uppercase tracking-widest border-b border-gray-50 dark:border-white/5">
                            <tr>
                                <th class="px-8 py-5">Categoría</th>
                                <th class="px-8 py-5 text-center">Qty</th>
                                <th class="px-8 py-5 text-right">Suma (S/)</th>
                                <th class="px-8 py-5 text-center">Peso</th>
                            </tr>
                        </thead>
                        <tbody class="text-sm divide-y divide-gray-50 dark:divide-white/5">
                            <?php foreach($catDensity as $cat): 
                                $weight = ($totalCapital > 0) ? ($cat['total_val'] / $totalCapital) * 100 : 0;
                            ?>
                            <tr class="hover:bg-gray-50 dark:hover:bg-white/5 transition-all">
                                <td class="px-8 py-5 font-bold text-gray-700 dark:text-gray-300"><?php echo $cat['name']; ?></td>
                                <td class="px-8 py-5 text-center font-mono"><?php echo $cat['qty']; ?></td>
                                <td class="px-8 py-5 text-right font-black text-gray-800 dark:text-white">S/ <?php echo number_format($cat['total_val'], 2); ?></td>
                                <td class="px-8 py-5">
                                    <div class="flex items-center gap-3 justify-center">
                                        <div class="w-16 bg-gray-100 dark:bg-white/10 h-1.5 rounded-full overflow-hidden">
                                            <div class="bg-brand-red h-full" style="width: <?php echo $weight; ?>%"></div>
                                        </div>
                                        <span class="text-[10px] font-black"><?php echo round($weight); ?>%</span>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>

        <div class="bg-white dark:bg-brand-dark rounded-[2.5rem] shadow-sm border border-gray-100 dark:border-white/5 overflow-hidden">
            <div class="p-8 border-b border-gray-50 dark:border-white/5">
                <h4 class="font-bold text-gray-800 dark:text-white uppercase text-xs tracking-widest">Últimas Adiciones al Catálogo</h4>
            </div>
            <div class="grid grid-cols-1 md:grid-cols-5 gap-0 divide-x divide-gray-50 dark:divide-white/5">
                <?php foreach($recent as $item): ?>
                <div class="p-6 hover:bg-gray-50 dark:hover:bg-white/5 transition-colors">
                    <p class="text-[10px] font-black text-brand-red uppercase mb-1"><?php echo $item['bname']; ?></p>
                    <h5 class="text-sm font-bold text-gray-800 dark:text-gray-200 truncate mb-2"><?php echo $item['name']; ?></h5>
                    <p class="text-lg font-display font-black dark:text-white">S/ <?php echo number_format($item['price'], 2); ?></p>
                    <p class="text-[9px] text-gray-400 mt-2 uppercase font-bold"><?php echo date('d M, Y', strtotime($item['created_at'])); ?></p>
                </div>
                <?php endforeach; ?>
            </div>
        </div>
    </main>

    <script>
        Chart.defaults.font.family = "'Barlow', sans-serif";
        Chart.defaults.color = '#9CA3AF';

        // Gráfico Radar: Distribución
        const ctxRadar = document.getElementById('radarBusiness').getContext('2d');
        new Chart(ctxRadar, {
            type: 'radar',
            data: {
                labels: <?php echo json_encode(array_column($catDensity, 'name')); ?>,
                datasets: [{
                    label: 'Modelos',
                    data: <?php echo json_encode(array_column($catDensity, 'qty')); ?>,
                    backgroundColor: 'rgba(217, 4, 41, 0.1)',
                    borderColor: '#D90429',
                    borderWidth: 2,
                    pointBackgroundColor: '#D90429',
                    pointRadius: 3
                }]
            },
            options: {
                responsive: true, maintainAspectRatio: false,
                scales: {
                    r: {
                        angleLines: { color: 'rgba(150,150,150,0.1)' },
                        grid: { color: 'rgba(150,150,150,0.1)' },
                        pointLabels: { font: { size: 9, weight: 'bold' } },
                        ticks: { display: false }
                    }
                },
                plugins: { legend: { display: false } }
            }
        });
    </script>

<?php include 'includes/footer.php'; ?>